
//------------------------------------------------------------------------------------------------
// Texture samplers
//------------------------------------------------------------------------------------------------

//@:-texformat blur_tex RGBA8

sampler2D blur_tex					: register(s3);	/// color tex downsampled and heavily blurred for bloom

//------------------------------------------------------------------------------------------------
// Shader constants
//------------------------------------------------------------------------------------------------

float4 thresholdPass		: register(c30);	///< { weightRed, weightGreen, weightBlue, thresholdLuminance }

/*#ifndef SM1

float4 bloomFactors			: register(c130);	///< { sceneBloomFactor, perFragmentGlowFactor, HDRrange, radialbloom }
float4 bloomRadian_params	: register(c136);	///< {minIntensity, intensityRange, startAngle, 1.0 / RangeAngle } // for horizontal gradiant
float4 bloomRadian_params2	: register(c137);	///< {minIntensity, intensityRange, startAngle, 1.0 / RangeAngle } // for spot gradiant
float4 bloomRadian_params3	: register(c138);	///< { worldLightDir.x, worldLightDir.w, worldLightDir.z, falloff }

#else*/

float4 bloomFactors			: register(c0);	///< { sceneBloomFactor, perFragmentGlowFactor, ?, ? }
float4 bloomRadian_params	: register(c1);	///< {minIntensity, intensityRange, startAngle, 1.0 / RangeAngle } // for horizontal gradiant
float4 bloomRadian_params2	: register(c2);	///< {minIntensity, intensityRange, startAngle, 1.0 / RangeAngle } // for spot gradiant
float4 bloomRadian_params3	: register(c3);	///< { worldLightDir.x, worldLightDir.w, worldLightDir.z, falloff }

//#endif


//------------------------------------------------------------------------------------------------
// Bloom
//------------------------------------------------------------------------------------------------

inline float4 Bloom( half4 srcColor, half2 texcoord, half4 bloomFactors, half2 noiseTexOffset )
{
	// bloomFactors.r => scene bloom factor
	// bloomFactors.b => source framebuffer range

	float4 bloomColor = tex2D(blur_tex, texcoord);

	// Scale the bloom color based on artist parameters
	float3 bloomColorAdjusted = bloomColor.rgb * bloomFactors.r;

#if !defined(SM1)
	// Subtract a little noise - this blends blocky transitions overall and
	// also lessens the visual extent of expansive and coarse dim regions.
//	float noise = tex2D(noiseTex, texcoord * float2(5.0, 2.8125) + noiseTexOffset).b * 0.007843; // Multiplier must be 2.0/255.0
//	bloomColorAdjusted = saturate(bloomColorAdjusted - noise); 
#endif

	// Extended range stored in source texture, expand to full range here and combine with bloom texture
//	float HDRRange = bloomFactors.b;
//	float3 dstColor = bloomColorAdjusted.rgb + (srcColor.rgb * HDRRange);

	// Conversion to LDR will happen in main filter, apply inverse to bloom so that we preserve maximum
	// quality of bloom texture (no point doubling it and losing one bit of accuracy)
	float3 dstColor = bloomColorAdjusted.rgb * 0.5 + srcColor.rgb;

	// force the output to be fully opaque
	return float4(dstColor, 1.0);
}

//_________________________________________________________________________________________________

/**
 * Compute bloom radian: the bloom intensity increase when we look up.
 */
inline half computeBloomRadian(VS_IN_2D input)
{
	// compute bloom custom strength: this is to support gradiant scene bloom (bloom strength depends on y value)
	
	// the vector from eye to vertex position
	half3 vector_eyeSpace;
	vector_eyeSpace.xy = g_FrustumParams.xz + ((input.position.xy * half2(0.5, 0.5)) + half2(0.5, 0.5)) * g_FrustumParams.yw; // xy in eye space
	vector_eyeSpace.z = g_ProjectionParams.x; // z in eye space

	// assuming the view matrix only contains a rotation (no scale), we can uses it to
	// transform from eye space to world space.
	vector_eyeSpace = normalize(vector_eyeSpace);
	half vector_worldSpace_y = dot(g_View[1].xyz, vector_eyeSpace);
	half alpha = vector_worldSpace_y * 0.5 + 0.5;
	
	// = minIntensity + saturate((alpha - startAngle) / rangeAngle) * intensityRange
	return bloomRadian_params.x + saturate((alpha - bloomRadian_params.z) * bloomRadian_params.w) * bloomRadian_params.y;
}

//_________________________________________________________________________________________________

/**
 * Compute bloom radian: the bloom intensity increase when we look along a specific direction
 * (simulate sun light spot effect: but where is the lens flare???).
 */
inline half computeBloomRadianSpot(VS_IN_2D input)
{
	// compute bloom custom strength: this is to support gradiant scene bloom (bloom strength depends on y value)
	
	// the vector from eye to vertex position
	half3 vector_eyeSpace;
	vector_eyeSpace.xy = g_FrustumParams.xz + (input.position.xy * 0.5 + 0.5) * g_FrustumParams.yw; // xy in eye space
	vector_eyeSpace.z = g_ProjectionParams.x; // z in eye space

	// assuming the view matrix only contains a rotation (no scale), we can uses it to
	// transform from eye space to world space.
	vector_eyeSpace = normalize(vector_eyeSpace);
	half3 vector_worldSpace = mul((half3x3)g_View, vector_eyeSpace);

	// compute angle between (the world light direction is in world space, normalized, from fragment to light)
	half dotprod = saturate(dot(vector_worldSpace, bloomRadian_params3.xyz));
	half angleBetween = acos(dotprod);

	// = minIntensity + saturate((alpha - startAngle) / rangeAngle) * intensityRange
	//                + (   1.0 - pow( saturate((angleBetween - startAngle) / rangeAngle), falloff )  ) * intensityRange
	
	half alpha = vector_worldSpace.y * 0.5 + 0.5;
	half horizontalRadian = saturate((alpha - bloomRadian_params.z) * bloomRadian_params.w) * bloomRadian_params.y;

	half C = saturate((angleBetween - bloomRadian_params2.z) * bloomRadian_params2.w);
	half spotRadian = (1.0 - pow(C, bloomRadian_params3.w)) * bloomRadian_params2.y;

	return bloomRadian_params.x + horizontalRadian + spotRadian;
}



